<?php

if (!defined('ABSPATH')) {
    exit;
}

final class GP_Admin
{
    private GP_Get_Plugin $plugin;
    private GP_Repository $repo;
    private GP_Access $access;
    private GP_Plugin_Manager $pm;

    public function __construct(GP_Get_Plugin $plugin, GP_Repository $repo, GP_Access $access, GP_Plugin_Manager $pm)
    {
        $this->plugin = $plugin;
        $this->repo   = $repo;
        $this->access = $access;
        $this->pm     = $pm;
    }

    public function init(): void
    {
        add_action('admin_menu', [$this, 'admin_menu']);
        add_action('admin_init', [$this, 'handle_actions']);
        add_action('admin_init', [$this, 'maybe_redirect_after_activation']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_assets']);
    }

    public function enqueue_assets(string $hook): void
    {
        $page = isset($_GET['page']) ? sanitize_text_field(wp_unslash($_GET['page'])) : '';
        if (!in_array($page, [GP_Get_Plugin::SLUG, GP_Get_Plugin::PAGE_NOT_INSTALLED], true)) {
            return;
        }

        wp_enqueue_style(
            'gp-get-plugin-admin',
            GP_GET_PLUGIN_URL . 'admin/assets/css/admin.css',
            [],
            GP_Get_Plugin::VERSION
        );

        wp_enqueue_script(
            'gp-get-plugin-admin',
            GP_GET_PLUGIN_URL . 'admin/assets/js/admin.js',
            [],
            GP_Get_Plugin::VERSION,
            true
        );
    }

    public function admin_menu(): void
    {
        $menu_title = gp_text('Get Plugin', 'دریافت افزونه');

        // bubble فقط وقتی سایت مجاز باشه + کاربر اجازه آپدیت داشته باشه
        $access = $this->access->check_access(false);
        if (!empty($access['allowed']) && current_user_can('update_plugins')) {
            $count = $this->pm->get_update_count();
            if ($count > 0) {
                $menu_title .= $this->updates_bubble_html($count);
            }
        }

        add_menu_page(
            gp_text('Get Plugin', 'دریافت افزونه'),
            $menu_title,
            'manage_options',
            GP_Get_Plugin::SLUG,
            [$this, 'render_installed_page'],
            'dashicons-admin-plugins',
            65
        );

        add_submenu_page(
            GP_Get_Plugin::SLUG,
            gp_text('Not Installed', 'نصب‌نشده‌ها'),
            gp_text('Not Installed', 'نصب‌نشده‌ها'),
            'manage_options',
            GP_Get_Plugin::PAGE_NOT_INSTALLED,
            [$this, 'render_not_installed_page']
        );

        // Rename first submenu (auto)
        global $submenu;
        if (isset($submenu[GP_Get_Plugin::SLUG][0][0])) {
            $submenu[GP_Get_Plugin::SLUG][0][0] = gp_text('Installed', 'نصب‌شده‌ها');
        }
    }

    private function updates_bubble_html(int $count): string
    {
        if ($count <= 0) return '';

        $display     = ($count > 999) ? '999+' : (string) number_format_i18n($count);
        $class_count = ($count > 999) ? 999 : $count;

        return ' <span class="update-plugins count-' . esc_attr((string) $class_count) . '"><span class="plugin-count">' . esc_html($display) . '</span></span>';
    }

    private function set_notice(string $type, string $message): void
    {
        $type = in_array($type, ['success', 'error', 'warning', 'info'], true) ? $type : 'info';

        set_transient(GP_Get_Plugin::NOTICE_KEY, [
            'type'    => $type,
            'message' => (string) $message,
            'ts'      => time(),
        ], 180);
    }

    private function get_notice(): ?array
    {
        $n = get_transient(GP_Get_Plugin::NOTICE_KEY);
        if (is_array($n)) {
            delete_transient(GP_Get_Plugin::NOTICE_KEY);
            return $n;
        }
        return null;
    }

    private function admin_page_url(string $page_slug): string
    {
        $allowed = [GP_Get_Plugin::SLUG, GP_Get_Plugin::PAGE_NOT_INSTALLED];
        if (!in_array($page_slug, $allowed, true)) {
            $page_slug = GP_Get_Plugin::SLUG;
        }
        return (string) admin_url('admin.php?page=' . rawurlencode($page_slug));
    }

    private function action_url(string $action, string $slug, string $ver, string $page_slug): string
    {
        $allowed_pages = [GP_Get_Plugin::SLUG, GP_Get_Plugin::PAGE_NOT_INSTALLED];
        if (!in_array($page_slug, $allowed_pages, true)) {
            $page_slug = GP_Get_Plugin::SLUG;
        }

        $url = (string) add_query_arg([
            'page'      => $page_slug,
            'gp_action' => $action,
            'gp_slug'   => $slug,
            'gp_ver'    => $ver,
        ], admin_url('admin.php'));

        $nonce_action = 'gp_' . $action . '|' . $slug . '|' . $ver . '|' . $page_slug;
        return wp_nonce_url($url, $nonce_action, '_gpnonce');
    }

    private function bulk_nonce_action(string $page_slug): string
    {
        return 'gp_bulk|' . $page_slug;
    }

    private function render_language_form(string $page): void
    {
        $current = GP_I18n::get_setting();
        ?>
        <div class="gp-langbar">
            <form method="post" action="<?php echo esc_url(admin_url('admin.php')); ?>" class="gp-lang-form">
                <input type="hidden" name="page" value="<?php echo esc_attr($page); ?>">
                <input type="hidden" name="gp_action" value="set_lang">
                <?php wp_nonce_field('gp_set_lang|' . $page, '_gpnonce_lang'); ?>

                <label for="gpPluginLang"><?php echo esc_html(gp_text('Plugin language:', 'زبان افزونه:')); ?></label>

                <select id="gpPluginLang" name="gp_plugin_lang">
                    <option value="auto" <?php selected($current, 'auto'); ?>>
                        <?php echo esc_html(gp_text('Auto (WordPress)', 'خودکار (زبان وردپرس)')); ?>
                    </option>
                    <option value="en" <?php selected($current, 'en'); ?>>English</option>
                    <option value="fa" <?php selected($current, 'fa'); ?>>فارسی</option>
                </select>

                <button type="submit" class="button button-secondary">
                    <?php echo esc_html(gp_text('Save', 'ذخیره')); ?>
                </button>
            </form>
        </div>
        <?php
    }

    public function render_installed_page(): void
    {
        $this->render_page('installed');
    }

    public function render_not_installed_page(): void
    {
        $this->render_page('not_installed');
    }

    public function handle_actions(): void
    {
        if (!is_admin()) return;

        require_once ABSPATH . 'wp-admin/includes/plugin.php';

        // ---------- set plugin language (allowed even if site not allowed) ----------
        if (
            ($_SERVER['REQUEST_METHOD'] ?? '') === 'POST'
            && isset($_POST['gp_action'])
            && sanitize_text_field(wp_unslash($_POST['gp_action'])) === 'set_lang'
        ) {
            $page = isset($_POST['page']) ? sanitize_text_field(wp_unslash($_POST['page'])) : GP_Get_Plugin::SLUG;
            if (!in_array($page, [GP_Get_Plugin::SLUG, GP_Get_Plugin::PAGE_NOT_INSTALLED], true)) {
                $page = GP_Get_Plugin::SLUG;
            }

            if (!current_user_can('manage_options')) {
                wp_die(esc_html__('Access denied.', 'get-plugin'));
            }

            $nonce = isset($_POST['_gpnonce_lang']) ? sanitize_text_field(wp_unslash($_POST['_gpnonce_lang'])) : '';
            if (!$nonce || !wp_verify_nonce($nonce, 'gp_set_lang|' . $page)) {
                $this->set_notice('error', gp_text('Nonce invalid. Please try again.', 'توکن امنیتی نامعتبر است. دوباره تلاش کنید.'));
                wp_safe_redirect($this->admin_page_url($page));
                exit;
            }

            $mode = isset($_POST['gp_plugin_lang']) ? sanitize_text_field(wp_unslash($_POST['gp_plugin_lang'])) : 'auto';
            GP_I18n::set_setting($mode);

            $this->set_notice('success', gp_text('Plugin language updated.', 'زبان افزونه تغییر کرد.'));
            wp_safe_redirect($this->admin_page_url($page));
            exit;
        }

        $access = $this->access->check_access(false);

        $req_action = isset($_REQUEST['gp_action']) ? sanitize_text_field(wp_unslash($_REQUEST['gp_action'])) : '';
        $is_trying_action = isset($_REQUEST['gp_action']) || isset($_REQUEST['gp_bulk_action']);
        $is_refresh = ($req_action === 'refresh');

        // اگر دسترسی ندارن: اکشن‌ها (به جز refresh) بلاک بشه
        if (empty($access['allowed']) && $is_trying_action && !$is_refresh) {
            $p = isset($_REQUEST['page']) ? sanitize_text_field(wp_unslash($_REQUEST['page'])) : '';
            $redirect_page = ($p === GP_Get_Plugin::PAGE_NOT_INSTALLED) ? GP_Get_Plugin::PAGE_NOT_INSTALLED : GP_Get_Plugin::SLUG;

            $site = isset($access['site']) && is_string($access['site'])
                ? (string) $access['site']
                : GP_Utils::normalize_url_for_compare(home_url('/'));

            $this->set_notice('error', sprintf(gp_text('Access is not allowed for %s.', 'دسترسی برای %s مجاز نیست.'), $site));
            wp_safe_redirect($this->admin_page_url($redirect_page));
            exit;
        }

        // ---------------- BULK (POST) ----------------
        if (isset($_POST['gp_bulk_action']) && ($_SERVER['REQUEST_METHOD'] ?? '') === 'POST') {
            $page = isset($_POST['page']) ? sanitize_text_field(wp_unslash($_POST['page'])) : GP_Get_Plugin::SLUG;
            if (!in_array($page, [GP_Get_Plugin::SLUG, GP_Get_Plugin::PAGE_NOT_INSTALLED], true)) {
                $page = GP_Get_Plugin::SLUG;
            }

            if (!current_user_can('manage_options')) {
                wp_die(esc_html__('Access denied.', 'get-plugin'));
            }

            $nonce = isset($_POST['_gpnonce_bulk']) ? sanitize_text_field(wp_unslash($_POST['_gpnonce_bulk'])) : '';
            if (!$nonce || !wp_verify_nonce($nonce, $this->bulk_nonce_action($page))) {
                $this->set_notice('error', gp_text('Nonce invalid. Please try again.', 'توکن امنیتی نامعتبر است. دوباره تلاش کنید.'));
                wp_safe_redirect($this->admin_page_url($page));
                exit;
            }

            $bulk_action = sanitize_text_field(wp_unslash($_POST['gp_bulk_action']));
            $selected = isset($_POST['gp_selected']) && is_array($_POST['gp_selected']) ? $_POST['gp_selected'] : [];

            $selected = array_values(array_unique(array_filter(array_map(static function ($s): string {
                return sanitize_text_field(wp_unslash((string) $s));
            }, $selected))));

            if (empty($selected)) {
                $this->set_notice('warning', gp_text('No plugins selected.', 'هیچ افزونه‌ای انتخاب نشده.'));
                wp_safe_redirect($this->admin_page_url($page));
                exit;
            }

            $allowed_bulk_installed = ['update', 'update_activate', 'activate', 'deactivate'];
            $allowed_bulk_not       = ['install', 'install_activate'];

            if ($page === GP_Get_Plugin::SLUG && !in_array($bulk_action, $allowed_bulk_installed, true)) {
                $this->set_notice('error', gp_text('Invalid bulk action.', 'عملیات دسته‌جمعی نامعتبر است.'));
                wp_safe_redirect($this->admin_page_url($page));
                exit;
            }
            if ($page === GP_Get_Plugin::PAGE_NOT_INSTALLED && !in_array($bulk_action, $allowed_bulk_not, true)) {
                $this->set_notice('error', gp_text('Invalid bulk action.', 'عملیات دسته‌جمعی نامعتبر است.'));
                wp_safe_redirect($this->admin_page_url($page));
                exit;
            }

            if (in_array($bulk_action, ['install', 'install_activate', 'update', 'update_activate'], true) && !current_user_can('install_plugins')) {
                $this->set_notice('error', gp_text('You do not have permission to install/update plugins.', 'شما اجازه نصب/بروزرسانی افزونه‌ها را ندارید.'));
                wp_safe_redirect($this->admin_page_url($page));
                exit;
            }
            if (in_array($bulk_action, ['activate', 'deactivate', 'install_activate', 'update_activate'], true) && !current_user_can('activate_plugins')) {
                $this->set_notice('error', gp_text('You do not have permission to activate/deactivate plugins.', 'شما اجازه فعال/غیرفعال کردن افزونه‌ها را ندارید.'));
                wp_safe_redirect($this->admin_page_url($page));
                exit;
            }

            $manifest = $this->repo->get_manifest_data(true, 25);
            if (!is_array($manifest) || !empty($manifest['error']) || empty($manifest['plugins']) || !is_array($manifest['plugins'])) {
                $err = isset($manifest['error']) ? (string) $manifest['error'] : 'unknown';
                $this->set_notice('error', gp_text('Manifest error: ', 'خطای مانیفست: ') . $err);
                wp_safe_redirect($this->admin_page_url($page));
                exit;
            }

            $remote_map = [];
            foreach ($manifest['plugins'] as $p) {
                if (!is_array($p)) continue;
                $s = isset($p['slug']) ? (string) $p['slug'] : '';
                if ($s !== '') $remote_map[$s] = $p;
            }

            $installed_map = $this->pm->installed_plugins_map();

            $ok = 0; $fail = 0; $skip = 0;
            $fail_msgs = [];

            foreach ($selected as $slug) {
                if (!isset($remote_map[$slug])) { $skip++; continue; }

                $remote      = $remote_map[$slug];
                $latest      = isset($remote['latest']) ? trim((string) $remote['latest']) : '';
                $remote_name = isset($remote['name']) ? (string) $remote['name'] : $slug;

                $is_installed = isset($installed_map[$slug]) && is_array($installed_map[$slug]) && !empty($installed_map[$slug]);

                // Not installed page
                if ($page === GP_Get_Plugin::PAGE_NOT_INSTALLED) {
                    if ($is_installed) { $skip++; continue; }
                    if ($latest === '') { $fail++; $fail_msgs[] = "$slug: latest empty"; continue; }

                    $activate_after = ($bulk_action === 'install_activate');
                    $res = $this->pm->install_or_update_one($slug, $latest, $remote, $activate_after, false);

                    if (!empty($res['ok'])) $ok++;
                    else { $fail++; $fail_msgs[] = "$slug: " . (isset($res['msg']) ? (string) $res['msg'] : 'failed'); }
                    continue;
                }

                // Installed page
                if (!$is_installed) { $skip++; continue; }

                $cands = $installed_map[$slug];
                $pf    = $this->pm->pick_installed_plugin_file($slug, $cands, $remote_name);

                if ($bulk_action === 'activate') {
                    if ($pf === '') { $fail++; $fail_msgs[] = "$slug: main file not found"; continue; }
                    $r = activate_plugin($pf);
                    if (is_wp_error($r)) { $fail++; $fail_msgs[] = "$slug: " . $r->get_error_message(); }
                    else $ok++;
                    continue;
                }

                if ($bulk_action === 'deactivate') {
                    if ($pf === '') { $fail++; $fail_msgs[] = "$slug: main file not found"; continue; }
                    deactivate_plugins($pf, false);
                    $ok++;
                    continue;
                }

                if ($latest === '') { $fail++; $fail_msgs[] = "$slug: latest empty"; continue; }

                $installed_ver = '';
                foreach ($cands as $c) {
                    if (!is_array($c)) continue;
                    if (isset($c['plugin_file']) && (string) $c['plugin_file'] === $pf) {
                        $installed_ver = isset($c['version']) ? (string) $c['version'] : '';
                        break;
                    }
                }

                if (!GP_Utils::is_update_available($installed_ver, $latest)) { $skip++; continue; }

                $activate_after  = ($bulk_action === 'update_activate');
                $preserve_active = !$activate_after;

                $res = $this->pm->install_or_update_one($slug, $latest, $remote, $activate_after, $preserve_active);

                if (!empty($res['ok'])) $ok++;
                else { $fail++; $fail_msgs[] = "$slug: " . (isset($res['msg']) ? (string) $res['msg'] : 'failed'); }
            }

            $this->plugin->invalidate_caches();

            $msg = sprintf(
                gp_text('Bulk done. Success: %s | Skipped: %s | Failed: %s', 'عملیات دسته‌جمعی انجام شد. موفق: %s | رد شد: %s | ناموفق: %s'),
                number_format_i18n($ok),
                number_format_i18n($skip),
                number_format_i18n($fail)
            );

            if ($fail > 0 && !empty($fail_msgs)) {
                $msg .= ' — ' . implode(' | ', array_slice($fail_msgs, 0, 4));
                if (count($fail_msgs) > 4) $msg .= ' ...';
                $this->set_notice('warning', $msg);
            } else {
                $this->set_notice('success', $msg);
            }

            wp_safe_redirect($this->admin_page_url($page));
            exit;
        }

        // ---------------- Single (GET) ----------------
        if (!isset($_GET['page'])) return;

        $page = sanitize_text_field(wp_unslash($_GET['page']));
        if (!in_array($page, [GP_Get_Plugin::SLUG, GP_Get_Plugin::PAGE_NOT_INSTALLED], true)) return;

        if (empty($_GET['gp_action']) || empty($_GET['gp_slug'])) return;

        $action = sanitize_text_field(wp_unslash($_GET['gp_action']));
        $slug   = sanitize_text_field(wp_unslash($_GET['gp_slug']));
        $ver    = isset($_GET['gp_ver']) ? sanitize_text_field(wp_unslash($_GET['gp_ver'])) : '';

        $allowed_actions = ['install', 'install_activate', 'update', 'update_activate', 'activate', 'deactivate', 'refresh'];
        if (!in_array($action, $allowed_actions, true)) return;

        $nonce_action = 'gp_' . $action . '|' . $slug . '|' . $ver . '|' . $page;
        if (!isset($_GET['_gpnonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['_gpnonce'])), $nonce_action)) {
            $this->set_notice('error', gp_text('Nonce invalid. Please try again.', 'توکن امنیتی نامعتبر است. دوباره تلاش کنید.'));
            wp_safe_redirect($this->admin_page_url($page));
            exit;
        }

        // refresh حتی بدون دسترسی هم مجازه
        if ($action === 'refresh') {
            $this->plugin->invalidate_caches();
            $this->set_notice('success', gp_text('Refreshed.', 'تازه‌سازی انجام شد.'));
            wp_safe_redirect($this->admin_page_url($page));
            exit;
        }

        if (in_array($action, ['install', 'install_activate', 'update', 'update_activate'], true) && !current_user_can('install_plugins')) {
            $this->set_notice('error', gp_text('You do not have permission to install plugins.', 'شما اجازه نصب افزونه را ندارید.'));
            wp_safe_redirect($this->admin_page_url($page));
            exit;
        }
        if (in_array($action, ['activate', 'deactivate', 'install_activate', 'update_activate'], true) && !current_user_can('activate_plugins')) {
            $this->set_notice('error', gp_text('You do not have permission to activate/deactivate plugins.', 'شما اجازه فعال/غیرفعال کردن افزونه را ندارید.'));
            wp_safe_redirect($this->admin_page_url($page));
            exit;
        }

        $manifest = $this->repo->get_manifest_data(true, 25);
        if (!is_array($manifest) || !empty($manifest['error']) || empty($manifest['plugins']) || !is_array($manifest['plugins'])) {
            $err = isset($manifest['error']) ? (string) $manifest['error'] : 'unknown';
            $this->set_notice('error', gp_text('Manifest error: ', 'خطای مانیفست: ') . $err);
            wp_safe_redirect($this->admin_page_url($page));
            exit;
        }

        $remote_map = [];
        foreach ($manifest['plugins'] as $p) {
            if (!is_array($p)) continue;
            $s = isset($p['slug']) ? (string) $p['slug'] : '';
            if ($s !== '') $remote_map[$s] = $p;
        }

        if (!isset($remote_map[$slug])) {
            $this->set_notice('error', gp_text('Plugin not found in manifest.', 'افزونه در مانیفست پیدا نشد.'));
            wp_safe_redirect($this->admin_page_url($page));
            exit;
        }

        $remote        = $remote_map[$slug];
        $remote_latest = isset($remote['latest']) ? (string) $remote['latest'] : '';
        if ($ver === '') $ver = $remote_latest;

        $remote_name = isset($remote['name']) ? (string) $remote['name'] : $slug;

        $installed_map = $this->pm->installed_plugins_map();
        $cands = isset($installed_map[$slug]) ? $installed_map[$slug] : [];
        $plugin_file = $this->pm->pick_installed_plugin_file($slug, is_array($cands) ? $cands : [], $remote_name);

        if ($action === 'activate') {
            if ($plugin_file === '') $this->set_notice('error', gp_text('Plugin not installed yet.', 'افزونه هنوز نصب نشده است.'));
            else {
                $r = activate_plugin($plugin_file);
                $this->set_notice(is_wp_error($r) ? 'error' : 'success', is_wp_error($r) ? $r->get_error_message() : gp_text('Plugin activated.', 'افزونه فعال شد.'));
            }
            $this->plugin->invalidate_caches();
            wp_safe_redirect($this->admin_page_url($page));
            exit;
        }

        if ($action === 'deactivate') {
            if ($plugin_file === '') $this->set_notice('error', gp_text('Plugin not installed yet.', 'افزونه هنوز نصب نشده است.'));
            else {
                deactivate_plugins($plugin_file, false);
                $this->set_notice('success', gp_text('Plugin deactivated.', 'افزونه غیرفعال شد.'));
            }
            $this->plugin->invalidate_caches();
            wp_safe_redirect($this->admin_page_url($page));
            exit;
        }

        if ($ver === '') {
            $this->set_notice('error', gp_text('Version is empty.', 'نسخه خالی است.'));
            wp_safe_redirect($this->admin_page_url($page));
            exit;
        }

        $activate_after  = in_array($action, ['install_activate', 'update_activate'], true);
        $preserve_active = ($action === 'update');

        $res = $this->pm->install_or_update_one($slug, $ver, $remote, $activate_after, $preserve_active);

        $this->plugin->invalidate_caches();

        if (!empty($res['ok'])) {
            $this->set_notice('success', $activate_after ? gp_text('Done (installed/updated & activated).', 'انجام شد (نصب/بروزرسانی و فعال شد).') : gp_text('Done.', 'انجام شد.'));
        } else {
            $this->set_notice('error', isset($res['msg']) ? (string) $res['msg'] : gp_text('Failed.', 'ناموفق.'));
        }

        wp_safe_redirect($this->admin_page_url($page));
        exit;
    }

    private function render_no_access(array $access, string $page): void
    {
        $site = isset($access['site']) ? (string) $access['site'] : GP_Utils::normalize_url_for_compare(home_url('/'));
        $err  = isset($access['error']) ? (string) $access['error'] : '';
        $src  = isset($access['source']) ? (string) $access['source'] : '';
        ?>
        <div class="wrap">
            <h1><?php echo esc_html(gp_text('Get Plugin', 'دریافت افزونه')); ?></h1>

            <?php $this->render_language_form($page); ?>

            <div class="notice notice-error">
                <p><strong><?php echo esc_html(sprintf(gp_text('Access is not allowed for %s.', 'دسترسی برای %s مجاز نیست.'), $site)); ?></strong></p>

                <?php if ($err !== '') : ?>
                    <p class="description" style="color:#646970;margin:8px 0 0;">
                        <?php echo esc_html(gp_text('Details:', 'جزئیات:')); ?>
                        <code><?php echo esc_html($err); ?></code>
                    </p>
                <?php endif; ?>

                <?php if ($src !== '') : ?>
                    <p class="description" style="color:#646970;margin:8px 0 0;">
                        <?php echo esc_html(gp_text('Allowed list source:', 'منبع لیست مجاز:')); ?>
                        <code><?php echo esc_html($src); ?></code>
                    </p>
                <?php endif; ?>
            </div>

            <p style="color:#646970;">
                <?php echo esc_html(gp_text(
                    'To get access, this site domain must be added to allowed_domains on the server.',
                    'برای دسترسی، دامنه این سایت باید داخل فایل allowed_domains روی سرور ثبت شده باشد.'
                )); ?>
            </p>

            <p>
                <a class="button button-secondary" href="<?php echo esc_url($this->admin_page_url($page)); ?>">
                    <?php echo esc_html(gp_text('Try again', 'تلاش دوباره')); ?>
                </a>
                <a class="button button-secondary" href="<?php echo esc_url($this->action_url('refresh', '__all__', '', $page)); ?>">
                    <?php echo esc_html(gp_text('Refresh', 'تازه‌سازی')); ?>
                </a>
            </p>
        </div>
        <?php
    }

    private function render_page(string $view): void
    {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('Access denied.', 'get-plugin'));
        }

        $page = isset($_GET['page']) ? sanitize_text_field(wp_unslash($_GET['page'])) : GP_Get_Plugin::SLUG;
        if (!in_array($page, [GP_Get_Plugin::SLUG, GP_Get_Plugin::PAGE_NOT_INSTALLED], true)) {
            $page = GP_Get_Plugin::SLUG;
        }

        // Access gate روی UI
        $access = $this->access->check_access(false);
        if (empty($access['allowed'])) {
            $this->render_no_access($access, $page);
            return;
        }

        $notice = $this->get_notice();

        $repo_url     = $this->repo->repo_url();
        $manifest_url = $this->repo->manifest_url();

        $manifest      = $this->repo->get_manifest_data(false, 15);
        $installed_map = $this->pm->installed_plugins_map();
        $update_count  = current_user_can('update_plugins') ? $this->pm->get_update_count() : 0;

        $installed_url     = $this->admin_page_url(GP_Get_Plugin::SLUG);
        $not_installed_url = $this->admin_page_url(GP_Get_Plugin::PAGE_NOT_INSTALLED);
        $refresh_url       = $this->action_url('refresh', '__all__', '', $page);

        ?>
        <div class="wrap">
            <h1><?php echo esc_html(gp_text('Get Plugin', 'دریافت افزونه')); ?></h1>

            <?php if (is_array($notice)) : ?>
                <div class="notice notice-<?php echo esc_attr((string) $notice['type']); ?> is-dismissible">
                    <p><?php echo esc_html((string) $notice['message']); ?></p>
                </div>
            <?php endif; ?>

            <?php if ($view === 'installed' && $update_count > 0 && current_user_can('update_plugins')) : ?>
                <div class="notice notice-warning">
                    <p><?php echo esc_html(sprintf(gp_text('⚠ %s installed plugin(s) need update.', '⚠ %s افزونه نصب‌شده نیاز به بروزرسانی دارد.'), number_format_i18n($update_count))); ?></p>
                </div>
            <?php endif; ?>

            <div class="gp-wrap">

                <div class="gp-meta">
                    <div><strong><?php echo esc_html(gp_text('Repo:', 'مخزن:')); ?></strong> <code><?php echo esc_html($repo_url); ?></code></div>
                </div>

                <?php $this->render_language_form($page); ?>

                <h2 class="nav-tab-wrapper gp-tabs">
                    <a href="<?php echo esc_url($installed_url); ?>" class="nav-tab <?php echo ($view === 'installed') ? 'nav-tab-active' : ''; ?>">
                        <?php echo esc_html(gp_text('Installed', 'نصب‌شده‌ها')); ?>
                    </a>
                    <a href="<?php echo esc_url($not_installed_url); ?>" class="nav-tab <?php echo ($view === 'not_installed') ? 'nav-tab-active' : ''; ?>">
                        <?php echo esc_html(gp_text('Not Installed', 'نصب‌نشده‌ها')); ?>
                    </a>
                </h2>

                <?php if (!is_array($manifest) || !empty($manifest['error'])) : ?>
                    <div class="notice notice-error" style="margin:14px 0 0;">
                        <p style="margin:8px 12px;">
                            <?php
                            $err = (!empty($manifest['error'])) ? (string) $manifest['error'] : 'Unknown error';
                            echo esc_html(gp_text('Error: ', 'خطا: ') . $err);
                            ?>
                        </p>
                    </div>
                <?php else : ?>

                    <?php
                    // ✅ فیلتر: فقط آیتم‌هایی که ورژن دارن نمایش داده بشن
                    $items_all = isset($manifest['plugins']) && is_array($manifest['plugins']) ? $manifest['plugins'] : [];
                    $items = [];

                    foreach ($items_all as $row) {
                        if (!is_array($row)) continue;

                        $slug = isset($row['slug']) ? (string) $row['slug'] : '';
                        if ($slug === '') continue;

                        // اگر نسخه ریموت خالی بود، نمایش نده
                        $latest = isset($row['latest']) ? trim((string) $row['latest']) : '';
                        if ($latest === '') continue;

                        $is_installed = isset($installed_map[$slug]) && is_array($installed_map[$slug]) && !empty($installed_map[$slug]);

                        if ($view === 'installed') {
                            if (!$is_installed) continue;

                            // اگر نسخه نصب‌شده خالی بود، نمایش نده
                            $name = isset($row['name']) ? (string) $row['name'] : $slug;
                            $cands = $installed_map[$slug];

                            $main = $this->pm->pick_installed_plugin_file($slug, is_array($cands) ? $cands : [], $name);
                            $installed_ver = '';

                            if ($main !== '' && is_array($cands)) {
                                foreach ($cands as $c) {
                                    if (!is_array($c)) continue;
                                    if (isset($c['plugin_file']) && (string) $c['plugin_file'] === $main) {
                                        $installed_ver = isset($c['version']) ? trim((string) $c['version']) : '';
                                        break;
                                    }
                                }
                            }

                            if ($installed_ver === '') continue;

                            $items[] = $row;
                            continue;
                        }

                        if ($view === 'not_installed') {
                            if ($is_installed) continue;
                            $items[] = $row;
                            continue;
                        }
                    }

                    $count = count($items);

                    // summary (installed)
                    $active_count    = 0;
                    $inactive_count  = 0;
                    $updates_in_list = 0;

                    if ($view === 'installed') {
                        foreach ($items as $r) {
                            if (!is_array($r)) continue;

                            $s = isset($r['slug']) ? (string) $r['slug'] : '';
                            if ($s === '') continue;

                            $nm = isset($r['name']) ? (string) $r['name'] : $s;
                            $lt = isset($r['latest']) ? (string) $r['latest'] : '';

                            $cands = isset($installed_map[$s]) ? $installed_map[$s] : [];
                            $main  = $this->pm->pick_installed_plugin_file($s, is_array($cands) ? $cands : [], $nm);
                            if ($main === '' || empty($cands)) continue;

                            $iv  = '';
                            $act = false;

                            foreach ($cands as $c) {
                                if (!is_array($c)) continue;
                                if (isset($c['plugin_file']) && (string) $c['plugin_file'] === $main) {
                                    $iv  = isset($c['version']) ? (string) $c['version'] : '';
                                    $act = !empty($c['active']);
                                    break;
                                }
                            }

                            if ($act) $active_count++;
                            else $inactive_count++;

                            if ($lt !== '' && GP_Utils::is_update_available($iv, $lt)) $updates_in_list++;
                        }
                    }
                    ?>

                    <form method="post" action="<?php echo esc_url(admin_url('admin.php')); ?>">
                        <input type="hidden" name="page" value="<?php echo esc_attr($page); ?>">
                        <?php wp_nonce_field($this->bulk_nonce_action($page), '_gpnonce_bulk'); ?>

                        <div class="gp-toolbar">
                            <div class="gp-left">
                                <input id="gpSearch" class="regular-text gp-search" type="search"
                                       placeholder="<?php echo esc_attr(gp_text('Search: name / slug ...', 'جستجو: نام / اسلاگ ...')); ?>" />

                                <?php if ($view === 'installed') : ?>
                                    <select id="gpFilter">
                                        <option value="all"><?php echo esc_html(gp_text('All', 'همه')); ?></option>
                                        <option value="updates"><?php echo esc_html(gp_text('With updates', 'فقط آپدیت‌دار')); ?></option>
                                        <option value="active"><?php echo esc_html(gp_text('Active', 'فعال')); ?></option>
                                        <option value="inactive"><?php echo esc_html(gp_text('Inactive', 'غیرفعال')); ?></option>
                                    </select>
                                <?php endif; ?>
                            </div>

                            <div class="gp-right">
                                <div class="gp-bulk">
                                    <select name="gp_bulk_action" required>
                                        <option value=""><?php echo esc_html(gp_text('Bulk actions…', 'کارهای دسته‌جمعی…')); ?></option>

                                        <?php if ($view === 'installed') : ?>
                                            <option value="update"><?php echo esc_html(gp_text('Update (selected)', 'بروزرسانی (انتخاب‌شده‌ها)')); ?></option>
                                            <option value="update_activate"><?php echo esc_html(gp_text('Update & Activate', 'بروزرسانی و فعال‌سازی')); ?></option>
                                            <option value="activate"><?php echo esc_html(gp_text('Activate', 'فعال‌سازی')); ?></option>
                                            <option value="deactivate"><?php echo esc_html(gp_text('Deactivate', 'غیرفعال‌سازی')); ?></option>
                                        <?php else : ?>
                                            <option value="install"><?php echo esc_html(gp_text('Install', 'نصب')); ?></option>
                                            <option value="install_activate"><?php echo esc_html(gp_text('Install & Activate', 'نصب و فعال‌سازی')); ?></option>
                                        <?php endif; ?>
                                    </select>

                                    <button type="submit" class="button action"><?php echo esc_html(gp_text('Apply', 'اعمال')); ?></button>
                                    <a class="button button-secondary" href="<?php echo esc_url($refresh_url); ?>"><?php echo esc_html(gp_text('Refresh', 'تازه‌سازی')); ?></a>
                                </div>
                            </div>
                        </div>

                        <div class="gp-summary">
                            <span class="gp-chip ghost">📦 <?php echo esc_html(number_format_i18n($count)); ?> <?php echo esc_html(gp_text('items', 'مورد')); ?></span>

                            <?php if ($view === 'installed') : ?>
                                <span class="gp-chip ok">✅ <?php echo esc_html(number_format_i18n($active_count)); ?> <?php echo esc_html(gp_text('active', 'فعال')); ?></span>
                                <span class="gp-chip ghost">⏸ <?php echo esc_html(number_format_i18n($inactive_count)); ?> <?php echo esc_html(gp_text('inactive', 'غیرفعال')); ?></span>
                                <span class="gp-chip warn">⬆ <?php echo esc_html(number_format_i18n($updates_in_list)); ?> <?php echo esc_html(gp_text('need update', 'آپدیت‌دار')); ?></span>
                            <?php endif; ?>

                            <?php if (!empty($manifest['generated_at'])) : ?>
                                <span class="gp-chip ghost">🕒 <?php echo esc_html((string) $manifest['generated_at']); ?></span>
                            <?php endif; ?>
                        </div>

                        <div class="gp-table-wrap">
                            <table class="widefat striped gp-table" id="gpTable">
                                <thead>
                                <tr>
                                    <th style="width:44px;"><input type="checkbox" id="gpSelectAll" /></th>
                                    <th><?php echo esc_html(gp_text('Plugin', 'افزونه')); ?></th>
                                    <th style="width:220px;"><?php echo esc_html(gp_text('Remote version', 'نسخه ریموت')); ?></th>

                                    <?php if ($view === 'installed') : ?>
                                        <th style="width:220px;"><?php echo esc_html(gp_text('Installed version', 'نسخه نصب‌شده')); ?></th>
                                        <th style="width:190px;"><?php echo esc_html(gp_text('Status', 'وضعیت')); ?></th>
                                    <?php endif; ?>

                                    <th style="width:320px;"><?php echo esc_html(gp_text('Actions', 'اقدامات')); ?></th>
                                </tr>
                                </thead>

                                <tbody>
                                <?php if ($count === 0) : ?>
                                    <tr>
                                        <td colspan="<?php echo ($view === 'installed') ? 6 : 4; ?>">
                                            <?php echo esc_html(gp_text('No items to display.', 'آیتمی برای نمایش وجود ندارد.')); ?>
                                        </td>
                                    </tr>
                                <?php else : ?>
                                    <?php foreach ($items as $row) : ?>
                                        <?php
                                        $slug   = isset($row['slug']) ? (string) $row['slug'] : '';
                                        $name   = isset($row['name']) ? (string) $row['name'] : $slug;
                                        $latest = isset($row['latest']) ? (string) $row['latest'] : '';
                                        $status = isset($row['status']) ? (string) $row['status'] : 'ok';

                                        $latest_ts = isset($row['latest_modified_ts']) ? (int) $row['latest_modified_ts'] : 0;
                                        $latest_h  = $latest_ts > 0 ? wp_date('Y-m-d H:i', $latest_ts) : '—';

                                        $row_text = strtolower($slug . ' ' . $name . ' ' . $latest);

                                        $installed_candidates = isset($installed_map[$slug]) ? $installed_map[$slug] : [];
                                        $installed_file = $this->pm->pick_installed_plugin_file($slug, is_array($installed_candidates) ? $installed_candidates : [], $name);

                                        $installed_ver = '';
                                        $is_active     = false;

                                        if ($installed_file !== '' && !empty($installed_candidates)) {
                                            foreach ($installed_candidates as $c) {
                                                if (!is_array($c)) continue;
                                                if (isset($c['plugin_file']) && (string) $c['plugin_file'] === $installed_file) {
                                                    $installed_ver = isset($c['version']) ? (string) $c['version'] : '';
                                                    $is_active     = !empty($c['active']);
                                                    break;
                                                }
                                            }
                                        }

                                        $is_installed = ($installed_file !== '');
                                        $has_update   = ($is_installed && $latest !== '' && GP_Utils::is_update_available($installed_ver, $latest));
                                        ?>
                                        <tr
                                            data-text="<?php echo esc_attr($row_text); ?>"
                                            <?php if ($view === 'installed') : ?>
                                                data-update="<?php echo esc_attr($has_update ? '1' : '0'); ?>"
                                                data-active="<?php echo esc_attr($is_active ? '1' : '0'); ?>"
                                            <?php endif; ?>
                                        >
                                            <td><input class="gp-select" type="checkbox" name="gp_selected[]" value="<?php echo esc_attr($slug); ?>"></td>

                                            <td>
                                                <div class="gp-plugin">
                                                    <span class="dashicons dashicons-admin-plugins gp-plugin__icon" aria-hidden="true"></span>
                                                    <div class="gp-plugin__body">
                                                        <div class="gp-plugin__title">
                                                            <span class="gp-plugin__name"><?php echo esc_html($name); ?></span>

                                                            <?php if ($view === 'installed') : ?>
                                                                <?php if ($is_active) : ?>
                                                                    <span class="gp-pill ok"><?php echo esc_html(gp_text('Active', 'فعال')); ?></span>
                                                                <?php else : ?>
                                                                    <span class="gp-pill"><?php echo esc_html(gp_text('Inactive', 'غیرفعال')); ?></span>
                                                                <?php endif; ?>

                                                                <?php if ($has_update) : ?>
                                                                    <span class="gp-pill warn"><?php echo esc_html(gp_text('Update available', 'آپدیت دارد')); ?></span>
                                                                <?php endif; ?>
                                                            <?php endif; ?>
                                                        </div>
                                                    </div>
                                                </div>
                                            </td>

                                            <td>
                                                <div><strong><?php echo esc_html($latest !== '' ? $latest : '—'); ?></strong></div>
                                                <div class="gp-muted" style="margin-top:4px;">
                                                    <?php echo esc_html(gp_text('Last modified:', 'آخرین تغییر:')); ?>
                                                    <?php echo esc_html($latest_h); ?>
                                                </div>
                                            </td>

                                            <?php if ($view === 'installed') : ?>
                                                <td>
                                                    <div><strong><?php echo esc_html($installed_ver !== '' ? $installed_ver : '—'); ?></strong></div>
                                                    <div style="margin-top:6px;">
                                                        <?php if ($is_active) : ?>
                                                            <span class="gp-badge ok"><?php echo esc_html(gp_text('✅ Active', '✅ فعال')); ?></span>
                                                        <?php else : ?>
                                                            <span class="gp-badge warn"><?php echo esc_html(gp_text('⏸ Inactive', '⏸ غیرفعال')); ?></span>
                                                        <?php endif; ?>
                                                    </div>
                                                </td>

                                                <td>
                                                    <?php if ($has_update) : ?>
                                                        <span class="gp-badge warn"><?php echo esc_html(gp_text('⬆ Needs update', '⬆ نیاز به بروزرسانی')); ?></span>
                                                    <?php else : ?>
                                                        <?php if ($status === 'ok') : ?>
                                                            <span class="gp-badge ok"><?php echo esc_html(gp_text('✅ Up to date', '✅ بروز')); ?></span>
                                                        <?php else : ?>
                                                            <span class="gp-badge warn"><?php echo esc_html($status); ?></span>
                                                        <?php endif; ?>
                                                    <?php endif; ?>
                                                </td>
                                            <?php endif; ?>

                                            <td>
                                                <div class="gp-actions">
                                                    <?php if ($view === 'not_installed') : ?>
                                                        <a class="button button-primary" href="<?php echo esc_url($this->action_url('install', $slug, $latest, $page)); ?>">
                                                            <?php echo esc_html(gp_text('Install', 'نصب')); ?>
                                                        </a>
                                                        <a class="button" href="<?php echo esc_url($this->action_url('install_activate', $slug, $latest, $page)); ?>">
                                                            <?php echo esc_html(gp_text('Install & Activate', 'نصب و فعال‌سازی')); ?>
                                                        </a>
                                                    <?php else : ?>
                                                        <?php if ($has_update) : ?>
                                                            <a class="button button-primary" href="<?php echo esc_url($this->action_url('update', $slug, $latest, $page)); ?>">
                                                                <?php echo esc_html(gp_text('Update', 'بروزرسانی')); ?>
                                                            </a>
                                                            <a class="button" href="<?php echo esc_url($this->action_url('update_activate', $slug, $latest, $page)); ?>">
                                                                <?php echo esc_html(gp_text('Update & Activate', 'بروزرسانی و فعال‌سازی')); ?>
                                                            </a>
                                                        <?php endif; ?>

                                                        <?php if ($is_installed) : ?>
                                                            <?php if (!$is_active) : ?>
                                                                <a class="button" href="<?php echo esc_url($this->action_url('activate', $slug, '', $page)); ?>">
                                                                    <?php echo esc_html(gp_text('Activate', 'فعال‌سازی')); ?>
                                                                </a>
                                                            <?php else : ?>
                                                                <a class="button" href="<?php echo esc_url($this->action_url('deactivate', $slug, '', $page)); ?>">
                                                                    <?php echo esc_html(gp_text('Deactivate', 'غیرفعال')); ?>
                                                                </a>
                                                            <?php endif; ?>
                                                        <?php endif; ?>
                                                    <?php endif; ?>
                                                </div>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                                </tbody>
                            </table>
                        </div>
                    </form>

                <?php endif; ?>
            </div>
        </div>
        <?php
    }

    public function maybe_redirect_after_activation(): void
    {
        if (!is_admin() || (function_exists('wp_doing_ajax') && wp_doing_ajax())) {
            return;
        }

        if (get_option('gp_just_activated') === '1') {
            delete_option('gp_just_activated');

            if (isset($_GET['activate-multi']) || (function_exists('is_network_admin') && is_network_admin())) {
                return;
            }

            wp_safe_redirect($this->admin_page_url(GP_Get_Plugin::SLUG));
            exit;
        }
    }
}
